import glob
from dag_core import check_param

def get_samples(fastqDir):
    SAMPLES = glob.glob('{}*.fq.gz'.format(fastqDir))
    SAMPLES = [sample.replace('{}'.format(fastqDir),'').replace('.fq.gz','') for sample in SAMPLES]
    SAMPLES = [('_').join(sample.split('_')[:-1])  for sample in SAMPLES]
    SAMPLES = list(set(SAMPLES))
    sample_file = open('.sample.list','w')
    for sample in SAMPLES:
        sample_file.write("%s\n" % sample)
    sample_file.close()
    return(SAMPLES)

def get_bwa_args(bwa_params):

    for param in ('seed_length', 'band_width', 'z_dropoff', 'mem_discard', 'match',  \
        'mismatch', 'gap_open', 'gap_extend', 'clipping_penalty', 'unpaired_penalty', \
        'low_threshold'):
        check_param(bwa_params,param,int)

    check_param(bwa_params,'rg_header',str)

    for param in ('rescue_only', 'output_all', 'hard_clipping', 'mark_split_secondary'):
        check_param(bwa_params,param,bool)

    check_param(bwa_params,'reseeding',float)

    bwa_args = "-k %i -w %i -d %i -r %f -c %i -A %i -B %i -O %i -E %i -L %i -U %i -T %i" \
        % (bwa_params['seed_length'],bwa_params['band_width'],bwa_params['z_dropoff'], \
        bwa_params['reseeding'], bwa_params['mem_discard'], bwa_params['match'], \
        bwa_params['mismatch'], bwa_params['gap_open'], bwa_params['gap_extend'], \
        bwa_params['clipping_penalty'], bwa_params['unpaired_penalty'], \
        bwa_params['low_threshold'])

    if bwa_params['rescue_only']:
        bwa_args+=" -P"

    if bwa_params['output_all']:
        bwa_args+=" -a"

    if bwa_params['hard_clipping']:
        bwa_args+=" -H"

    if bwa_params['mark_split_secondary']:
        bwa_args+=" -M"

    if len(bwa_params['rg_header']):
        assert re.match('^RG:',bwa_params['rg_header']), \
            'Invalid read group header provided: %s' % bwa_params['rg_header']
        bwa_args+=' -R %s' % bwa_params['rg_header']

    return(bwa_args)

def get_markdup_args(markdup_params):

	for param in ('comment', 'read_name_regex'):
		check_param(markdup_params, param, str)

	for param in ('optical_set_size', 'min_distance', 'optical_pixel_distance'):
		check_param(markdup_params, param, int)

	for param in ('remove_duplicates', 'skip_no_mate'):
		check_param(markdup_params, param, bool)

	strategies=('TOTAL_MAPPED_REFERENCE_LENGTH','SUM_OF_BASE_QUALITIES','RANDOM')
	assert markdup_params['scoring_strategy'] in strategies, \
		'Invalid scoring strategy encoding specified: %s' % markdup_parms['scoring_strategy']

	programs=('MarkDuplicates','MarkDuplicatesWithMateCigar')
	assert markdup_params['program'] in programs, \
		'Invalid program specified: %s' % markdup_params['program']

	# Temporary option while picard migrate argument conventions...
	markdup_args = '-Dpicard.useLegacyParser=false'

	markdup_args += ' --MAX_OPTICAL_DUPLICATE_SET_SIZE %i --OPTICAL_DUPLICATE_PIXEL_DISTANCE %i --DUPLICATE_SCORING_STRATEGY %s' \
		% (markdup_params['optical_set_size'],  markdup_params['optical_pixel_distance'], \
	 	markdup_params['scoring_strategy'])

	if markdup_params['comment']:
		markdup_args += " --COMMENT %s" % markdup_params['comment']

	if markdup_params['read_name_regex']:
		markdup_args += " --READ_NAME_REGEX %s" % markdup_params['read_name_regex']

	if markdup_params['remove_duplicates']:
		markdup_args += " --REMOVE_DUPLICATES true"

	if markdup_params['program'] == 'MarkDuplicateWithMateCigar':
		if markdup_params['skip_no_mate']:
			markdup_args += " --SKIP_PAIRS_WITH_NO_MATE_CIGAR true"
		markdup_args += " --MINIMUM_DISTANCE %i" % markdup_params['min_distance']

	markdup_dat={
		'program':markdup_params['program'],
		'args': markdup_args
	}

	return(markdup_dat)

def get_metrics_args(metrics_params):


	for param in ('as:max_insert_size', 'gc:window_size', 'oxoG:min_qual', 'oxoG:min_mapq', 'oxoG:min_inssize', 
			'oxoG:max_inssize', 'oxoG:context_size', 'wgs:cov_cap', 'wgs:locus_cap', 'wgs:min_qual', 
			'wgs:min_mapq', 'wgs:read_length', 'wgs:sample_size'):
		check_param(metrics_params, param, int)

	# histogram width should be an int if set, but defaults to 'null'
	# so defined as 'None' in Snakefile by default
	if metrics_params['is:histogram_width'] is not None:
		check_param(metrics_params, 'is:histogram_width', int)

	for param in ('gc:ignore_duplicates', 'is:include_duplicates', 'oxoG:non_pf_reads', 'oxoG:use_oq', 
	'bd:aligned_only', 'bd:pf_reads_only', 'wgs:count_unpaired', 'wgs:bq_histo'):
		check_param(metrics_params, param, bool)

	for param in ('gc:min_genome_fraction', 'is:deviations', 'is:minimum_pct'):
		check_param(metrics_params, param, float)

	check_param(metrics_params,'as:adapter_sequence', str)
	if metrics_params['oxoG:intervals'] is not None:
		check_param(metrics_params, 'oxoG:intervals', str)
	if metrics_params['oxoG:dbsnp'] is not None:
		check_param(metrics_params, 'oxoG:dbsnp', str)
	
	orientations=('FR','RF','TANDEM')
	assert metrics_params['as:pair_orientation'] in orientations, \
		'Invalid pair orientation specified: %s' % metrics_params['as:pair_orientation']

	levels=('ALL_READS','SAMPLE','LIBRARY','READ_GROUP')
	assert metrics_params['accumulation_level'] in levels, \
		'Invalid accumulation level specified: %s' % metrics_params['accumulation_level']
	
	# Alignment summary
	alignmentSummary_args = " --MAX_INSERT_SIZE %i --EXPECTED_PAIR_ORIENTATIONS %s --METRIC_ACCUMULATION_LEVEL %s" % \
		(metrics_params['as:max_insert_size'], metrics_params['as:pair_orientation'], metrics_params['accumulation_level'])

	if metrics_params['as:adapter_sequence']:
		alignmentSummary_args += " --ADAPTER_SEQUENCE %s" % metrics_params['as:adapter_sequence']

	# Base Distribution
	baseDistribution_args = ""
	if metrics_params['bd:aligned_only']:
		baseDistribution_args += " --ALIGNED_READS_ONLY"
	if metrics_params['bd:pf_reads_only']:
		baseDistribution_args += " --PF_READS_ONLY"

	# GC Bias
	gcBias_args = " --METRIC_ACCUMULATION_LEVEL %s --SCAN_WINDOW_SIZE %i --MINIMUM_GENOME_FRACTION %f" % \
		(metrics_params['accumulation_level'], metrics_params['gc:window_size'], metrics_params['gc:min_genome_fraction'])
	
	if metrics_params['gc:ignore_duplicates']:
		gcBias_args += ' --ALSO_IGNORE_DUPLICATES true'

	# InsertSize
	insertSize_args = " --DEVIATIONS %f --MINIMUM_PCT %f" % \
		(metrics_params['is:deviations'],metrics_params['is:minimum_pct'], )
	if metrics_params['is:histogram_width'] is not None:
		insertSize_args += ' --HISTOGRAM_WIDTH %i' % metrics_params['is:histogram_width']
	if metrics_params['is:include_duplicates']:
		insertSize_args += ' --INCLUDE_DUPLICATES'

	#oxoG
	oxoG_args = " --MINIMUM_QUALITY_SCORE %i --MINIMUM_MAPPING_QUALITY %i --MINIMUM_INSERT_SIZE %i --MAXIMUM_INSERT_SIZE %i \
			--CONTEXT_SIZE %i" % \
		(metrics_params['oxoG:min_qual'], metrics_params['oxoG:min_mapq'], metrics_params['oxoG:min_inssize'], 
		metrics_params['oxoG:max_inssize'], metrics_params['oxoG:context_size'])

	if metrics_params['oxoG:non_pf_reads']:
		oxoG_args += " --INCLUDE_NON_PF_READS"
	if metrics_params['oxoG:use_oq']:
		oxoG_args += " --USE_OQ"
	if metrics_params['oxoG:intervals'] is not None:
		oxoG_args += " --INTERVALS %s" % metrics_params['oxoG:intervals']
	if metrics_params['oxoG:dbsnp'] is not None:
		oxoG_args += " --DB_SNP %s" % metrics_params['oxoG:dbsnp']

	# wgs metrics
	wgs_args = " --COVERAGE_CAP %i --LOCUS_ACCUMULATION_CAP %i --MINIMUM_BASE_QUALITY %i \
		--READ_LENGTH %i --SAMPLE_SIZE %i" % \
		(metrics_params['wgs:cov_cap'], metrics_params['wgs:locus_cap'], metrics_params['wgs:min_mapq'],
		metrics_params['wgs:read_length'], metrics_params['wgs:sample_size']) 
	if metrics_params['wgs:count_unpaired']:
		wgs_args += " --COUNT_UNPAIRED"
	if metrics_params['wgs:bq_histo']:
		wgs_args += " --INCLUDE_BQ_HISTOGRAM"
	
	metrics_args={
		'alignment_summary': alignmentSummary_args,
		'base_distribution': baseDistribution_args,
		'gc_bias': gcBias_args,
		'insert_size': insertSize_args,
		'oxoG': oxoG_args,
		'wgs': wgs_args
	}

	return(metrics_args)

def get_multiqc_args(mqc_params):
    
    plot_types=('default','flat','interactive')
    assert mqc_params['plot_type'] in plot_types, \
        'Invalid plot_type specified: %s' % mqc_parms['encoding']
    
    multiqc_args=""
    
    if mqc_params['plot_type'] == 'flat':
        multiqc_args += '--flat'
    elif mqc_params['plot_type'] == 'interactive':
        multiqc_args += '--interactive'
    
    return(multiqc_args)

