#!/usr/bin/env python

import subprocess
import os
import pickle
import re
import sys 
from datetime import datetime
from jinja2 import Environment, FileSystemLoader, select_autoescape
from dag_core import export_environment

def days_hours_minutes(td):
    return td.days, td.seconds//3600, (td.seconds//60)%60

def get_versions():

    bwa_ps = subprocess.Popen(('bwa'),stderr=subprocess.PIPE)
    bwa_version = subprocess.check_output(('grep', 'Version'), stdin=bwa_ps.stderr)
    bwa_ps.wait() 
    picard_version = subprocess.run(['picard', 'BamIndexStats', '--version'],stderr=subprocess.PIPE)
    multiqc_version = subprocess.run(['multiqc','--version'],stdout=subprocess.PIPE)
    
    citations = {
        'BWA': 'Li H. (2013) Aligning sequence reads, clone sequences and assembly contigs with BWA-MEM. arXiv:1303.3997v2i',
        'Picard': 'http://broadinstitute.github.io/picard',
        'MultiQC': 'Ewels, P. et al. (2016). MultiQC: summarize analysis results for multiple tools and samples in a single report. Bioinformatics, 32(19)  pp. 3047-3048, doi:https://doi.org/10.1093/bioinformatics/btw354',
    }
    
    p=re.compile('Version: ([\d\.\-r]+)')
    m=p.search(bwa_version.decode('utf-8'))
    if m:
        bwa_version=m.group(1)
    else:
        print("Error parsing BWA version number")
        sys.exit(1)
     
    p=re.compile('version\s([\d\.]+)')
    m=p.search(multiqc_version.stdout.decode('utf-8'))
    if m:
        multiqc_version = m.group(1)
    else:
        print("Error parsing multiqc version number")
        sys.exit(1)
     
    versions = {
        'BWA': {'version':bwa_version, 'citation': citations['BWA']},
        'Picard': {'version':picard_version.stderr.decode('utf-8'),'citation':citations['Picard']},
        'MultiQC': {'version':multiqc_version, 'citation': citations['MultiQC']},
    }
    return(versions)

report_file = sys.argv[1]

start_file=open(".start_date.txt", 'r')
start_date=start_file.readline().rstrip("\n")
start_file.close()
start_datetime=datetime.strptime(start_date,'%a %b %d %Y %X')
end_date=datetime.now()
runtime=end_date-start_datetime

jobname_file=open(".jobname",'r')
jobname=jobname_file.readline().rstrip("\n")
jobname_file.close()

args_file=open(".bwaparams","rb")
params=args_file.read()
args_file.close()
bwa_params=pickle.loads(params)

args_file=open(".markdup_params","rb")
params=args_file.read()
args_file.close()
md_params=pickle.loads(params)

SAMPLES=[]
with open(".sample.list") as f:
	for line in f:
		line=line.rstrip("\n")
		SAMPLES.append(line)
f.close()
SAMPLES.sort()

sample_summary=[]
for sample in SAMPLES:
	sample_data={
		'name':sample,
	}
	count_re=re.compile('^([0-9]+)')
	flagstat_handle=open("flagstats/%s.flagstat.txt" % sample,'r') 
	flagstats=flagstat_handle.readlines()
	m=count_re.search(flagstats[0])
	if m:	
		sample_data['total_reads']=m.group(1)

	m=count_re.search(flagstats[3])
	if m:	
		sample_data['duplicates']=m.group(1)

	m=count_re.search(flagstats[4])
	if m:	
		sample_data['mapped']=m.group(1)

	m=count_re.search(flagstats[8])
	if m:	
		sample_data['paired']=m.group(1)

	m=count_re.search(flagstats[10])
	if m:	
		sample_data['singletons']=m.group(1)

	sample_summary.append(sample_data)

versions=get_versions()
export_environment()
 
report=open(report_file,'w')

cp=os.environ['CONDA_PREFIX']
template_dir=("%s/etc/dag-wf/map-bwa-mem-pe/" % cp)
env = Environment(
	loader=FileSystemLoader(template_dir), autoescape=select_autoescape(['html'])
)
template = env.get_template('report.tmpl')

html=template.render(
		jobname=jobname, 
		start_date=start_date, 
		end_date=end_date.strftime("%a %b %d %Y %X"), 
		run_time=runtime,
		software=versions,
		sample_data=sample_summary,
		bwa_seed_length=bwa_params['seed_length'],
		bwa_bandwidth=bwa_params['band_width'],
		bwa_z_dropoff=bwa_params['z_dropoff'],
		bwa_reseeding=bwa_params['reseeding'],
		bwa_mem_discard=bwa_params['mem_discard'],
		bwa_rescue_only=bwa_params['rescue_only'],
		bwa_match=bwa_params['match'],
		bwa_mismatch=bwa_params['mismatch'],
		bwa_gap_open=bwa_params['gap_open'],
		bwa_gap_extend=bwa_params['gap_extend'],
		bwa_clipping_penalty=bwa_params['clipping_penalty'],
		bwa_unpaired_penalty=bwa_params['unpaired_penalty'],
		bwa_low_threshold=bwa_params['low_threshold'],
		bwa_output_all=bwa_params['output_all'],
		bwa_hard_clipping=bwa_params['hard_clipping'],
		bwa_mark_split_secondary=bwa_params['mark_split_secondary'],
		md_program=md_params['program'],
		md_scoring_strategy=md_params['scoring_strategy'],
		md_optical_set_size=md_params['optical_set_size'],
		md_min_distance=md_params['min_distance'],
		md_optical_pixel_distance=md_params['optical_pixel_distance'],
		md_read_name_regex=md_params['read_name_regex'],
		md_remove_duplicates=md_params['remove_duplicates'],
		md_skip_no_mate=md_params['skip_no_mate']	
)

report.write(html)
                
report.close()

