""" Utility functions common to various scripts """
import re
import sys
import os
import glob
import readchar
import drmaa

from Bio import SeqIO

def check_format(file: str):
    """Identify format of sequence file

    Required Parameters:
        file: path to file to identify format of 

    Returns:
        type: string containing 'genbank', 'embl' or 'fasta'
    """

    file=open(file, 'r')
    line=file.readline()
    gb=re.match('LOCUS',line)
    embl=re.match('ID',line)
    fasta=re.match('>',line)

    if gb:
        return('genbank')
    elif embl:
        return('embl')
    elif fasta:
        return('fasta')
    else:
        sys.exit('Error: could not identify format of {}'.format(file))

def clobber_check(path):
    """Checks if a file already exists prior to overwriting it
       Exits if user chooses not to overwrite...
    """
    if os.path.isfile(path):
        print("{} already exists. Overwrite? [y/N]".format(path))
        response = readchar.readchar()
        if response == 'y' or response == 'Y':
            return()
        else:
            print("Aborting...")
            sys.exit(1)

def to_fasta(file,file_format):
    """Convert embl/genbank record to fasta"""

    pre=os.path.splitext(file)[0]
    output='{}.fa'.format(pre)

    records=[]
    for record in SeqIO.parse(file, file_format):
        records.append(record)

    clobber_check(output)

    SeqIO.write(records, output, 'fasta')

    return(output)

def drmaa_run(tmpdir: str, scripts: list, job_name: str):
    """ Runs provided script via drmaa

    Required params: 
        tmpdir: path to temporary directory
        script: list of strings - path to scripts to run
        job_name: Name of job

    Returns:
        None
    """
    with drmaa.Session() as s:

        job_ids=[]
        job_templates=[]

        env={
        'HOME': os.environ['HOME'],
        'SHELL': os.environ['SHELL'],
        'PATH': os.environ['PATH'],
        'SGE_CELL': os.environ['SGE_CELL'],
        'SGE_EXECD_PORT': os.environ['SGE_EXECD_PORT'],
        'SGE_QMASTER_PORT': os.environ['SGE_QMASTER_PORT'],
        'SGE_ROOT': os.environ['SGE_ROOT'],
        'SGE_CLUSTER_NAME': os.environ['SGE_CLUSTER_NAME'],
        'DRMAA_LIBRARY_PATH': os.environ['DRMAA_LIBRARY_PATH']
        }

        for script in scripts:
            jt = s.createJobTemplate()
            jt.remoteCommand = os.path.join(os.getcwd(), script)
            jt.workingDirectory=os.getcwd()
            jt.joinFiles=True
            jt.jobName=job_name
            jt.jobEnvironment=env
            jt.nativeSpecification='-pe smp 8'

            job_id = s.runJob(jt)
            job_ids.append(job_id)
        
        print('{} jobs submitted: {}'.format(job_name,','.join(job_ids)) )
        s.synchronize(job_ids, drmaa.Session.TIMEOUT_WAIT_FOREVER, False)

        for job_id in job_ids:
            retval=s.wait(job_id,drmaa.Session.TIMEOUT_WAIT_FOREVER)
            if retval.exitStatus != 0:
                print("Job {} produced non-zero exit value: {}".format(job_id,retval.exitStatus))
                sys.exit(1)

        logs=glob.glob('{}*'.format(job_name))
        for log in logs:
            os.remove(log)

        for jt in job_templates:
            s.deleteJobTemplate(jt)

        for script in scripts:
            os.remove(script)
