#!/usr/bin/env python

"""Generates a blast comparison suitable for loading 
    into ACT
"""

import argparse
import re
import sys
import os
from utils import check_format,to_fasta,clobber_check

from Bio import SeqIO
from Bio.Blast.Applications import NcbiblastnCommandline

def compare(args):
    """Create comparison..."""

    dbs=[]
    for file in (args.reference, args.subject):
        genome_format=check_format(file=file)
        if genome_format != 'fasta':
            filename=to_fasta(file=file,file_format=genome_format)
        else:
            filename=file
        dbs.append(filename)

    # use the directory containing the subject as the output path
    out_path=os.path.dirname(dbs[1])
    ref_base=os.path.basename(dbs[0])
    ref_base=os.path.splitext(ref_base)[0]

    subject_base=os.path.basename(dbs[1])
    subject_base=os.path.splitext(subject_base)[0]

    out_file="{}_vs_{}.blastn".format(subject_base,ref_base)
    if out_path:
        out_file="{}/{}".format(out_path,out_file)

    clobber_check(out_file)

    print("\nWriting BLAST output to {}\n".format(out_file))

    blast_cline = NcbiblastnCommandline(query=dbs[1], subject=dbs[0], evalue=0.001, outfmt=6, out=out_file)
    stderr=blast_cline()[1]
    
    if stderr:
        print("Warning - the following STDERR was reported by BLAST")
        print(stderr)

    os.remove(dbs[0])
    os.remove(dbs[1])

if __name__ == "__main__":
    parser = argparse.ArgumentParser(
        description="Creates a comparison between two genome records suitable for loading into ACT")

    parser.add_argument('--reference', dest='reference', 
        help='path to EMBL/Genbank/Fasta file containing reference genome sequence', required=True)
    parser.add_argument('--subject', dest='subject', 
        help='path to EMBL/Genbank/Fasta file containing subject genome sequence', required=True)

    args = parser.parse_args()

    compare(args)